#!/bin/bash

# SAAM Cabins - Deployment Setup Script
echo "🚀 SAAM Cabins Deployment Setup"
echo "================================="

# Create deployment directory
mkdir -p deployment
mkdir -p deployment/admin

# Copy built files to deployment directory
echo "📦 Copying built files..."
cp -r dist/* deployment/admin/

# Create .htaccess for Apache (if needed)
cat > deployment/admin/.htaccess << 'EOF'
# Enable compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/css text/javascript application/javascript application/x-javascript text/xml application/xml application/xml+rss text/plain
</IfModule>

# Cache static assets
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresByType text/css "access plus 1 year"
    ExpiresByType application/javascript "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
</IfModule>

# Security headers
Header always set X-Frame-Options "SAMEORIGIN"
Header always set X-Content-Type-Options "nosniff"
Header always set Referrer-Policy "strict-origin-when-cross-origin"

# Protect sensitive files
<FilesMatch "\.(json|env|log)$">
    Require all denied
</FilesMatch>
EOF

# Create environment template
cat > deployment/admin/.env.local.example << 'EOF'
# SAAM Cabins Environment Configuration
NEXTAUTH_SECRET=your-very-secure-secret-key-here
NEXTAUTH_URL=https://admin.yourdomain.com
UPLOAD_DIR=/public_html/admin/uploads
NODE_ENV=production
EOF

# Create deployment instructions
cat > deployment/UPLOAD-INSTRUCTIONS.txt << 'EOF'
SAAM CABINS - UPLOAD INSTRUCTIONS
=================================

1. CREATE SUBDOMAIN IN CPANEL:
   - Go to cPanel → Subdomains
   - Create subdomain: "admin"
   - Document root: /public_html/admin

2. UPLOAD FILES:
   - Upload ALL contents of the 'admin' folder to /public_html/admin/
   - Maintain folder structure exactly as shown

3. SET PERMISSIONS:
   - Folders: 755
   - Files: 644
   - uploads/ folder: 755
   - data/ folder: 755

4. CONFIGURE ENVIRONMENT:
   - Copy .env.local.example to .env.local
   - Update with your domain details

5. TEST ACCESS:
   - Main site: https://admin.yourdomain.com
   - Admin panel: https://admin.yourdomain.com/admin/login
   - Default login: admin / admin123

6. SECURITY:
   - Change default admin password immediately!
   - Install SSL certificate
   - Regular backups

For detailed instructions, see DEPLOYMENT-GUIDE.md
EOF

# Create file structure documentation
cat > deployment/FILE-STRUCTURE.txt << 'EOF'
UPLOAD FILE STRUCTURE
====================

Upload the 'admin' folder contents to your subdomain directory:

/public_html/admin/
├── _next/                  (Next.js assets)
├── admin/                  (Admin panel pages)
│   ├── dashboard/
│   ├── login/
│   ├── company/
│   ├── sliders/
│   └── pages/
├── api/                    (API endpoints)
│   └── admin/
├── about/                  (About page)
├── contact/                (Contact page)
├── products/               (Product pages)
├── work/                   (Work portfolio)
├── uploads/                (Image uploads - ensure writable)
├── data/                   (CMS data - ensure writable)
├── index.html              (Homepage)
├── 404.html                (Error page)
├── .htaccess               (Apache config)
└── .env.local              (Environment config)

IMPORTANT FOLDERS TO MAKE WRITABLE:
- uploads/ (755 permissions)
- data/ (755 permissions)
EOF

# Create admin credentials file
cat > deployment/ADMIN-CREDENTIALS.txt << 'EOF'
SAAM CABINS - ADMIN ACCESS
=========================

DEFAULT CREDENTIALS:
Username: admin
Password: admin123

ADMIN PANEL URLs:
- Login: https://admin.yourdomain.com/admin/login
- Dashboard: https://admin.yourdomain.com/admin/dashboard
- Company: https://admin.yourdomain.com/admin/company
- Products: https://admin.yourdomain.com/admin/products
- Sliders: https://admin.yourdomain.com/admin/sliders
- Pages: https://admin.yourdomain.com/admin/pages

⚠️  SECURITY WARNING:
Change the default password immediately after first login!

CMS FEATURES:
✓ Company information management
✓ Product catalog with images
✓ Dynamic slider management
✓ Page builder
✓ Testimonial management
✓ Image upload system
✓ SEO optimization
EOF

echo "✅ Deployment package ready!"
echo ""
echo "📁 Files prepared in 'deployment' folder:"
echo "   - admin/ (all website files)"
echo "   - .htaccess (server configuration)"
echo "   - .env.local.example (environment template)"
echo "   - UPLOAD-INSTRUCTIONS.txt"
echo "   - FILE-STRUCTURE.txt"
echo "   - ADMIN-CREDENTIALS.txt"
echo ""
echo "🚀 Ready to upload to cPanel!"
echo "📖 See UPLOAD-INSTRUCTIONS.txt for detailed steps"